/**
 * OWASP AppSensor
 * 
 * This file is part of the Open Web Application Security Project (OWASP)
 * AppSensor project. For details, please see
 * <a href="http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project">
 * 	http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project</a>.
 *
 * Copyright (c) 2010 - The OWASP Foundation
 * 
 * AppSensor is published by OWASP under the BSD license. You should read and accept the
 * LICENSE before you use, modify, and/or redistribute this software.
 * 
 * @author Michael Coates <a href="http://www.aspectsecurity.com">Aspect Security</a>
 * @author John Melton <a href="http://www.jtmelton.com/">jtmelton</a>
 * @created 2010
 */
namespace org.owasp.appsensor
{
    using System;
    using System.Collections.Generic;
    using System.Configuration;
    using Owasp.Esapi;
    using Owasp.Esapi.Errors;
    using AppSensor2.configuration;

    /**
     * This class simply : the DefaultSecurityConfiguration from Esapi
     * in order to ensure we can recognize a few extra configuration options 
     * that are specific to AppSensor.  These include the configuration for 
     * the response action and intrusion store implementations as well as 
     * the configuration of the duration and timescale fields for disabling
     * components.  Other than that, everything from the DefaultSecurityConfiguration
     * still exists.   
     * 
     * 9/14/2010 - jtm - had to add singleton config since esapi 2.0rc7 changed their 
     * management code for instantiation of singletons. This is related to bug: 
     * http://code.google.com/p/appsensor/issues/detail?id=3
     * 
     * @author Michael Coates (michael.coates .at. owasp.org) 
     *         <a href="http://www.aspectsecurity.com">Aspect Security</a>
     * @author John Melton (jtmelton .at. gmail.com)
     *         <a href="http://www.jtmelton.com/">jtmelton</a>
     * @since February 24, 2010
     * @see org.owasp.esapi.reference.DefaultSecurityConfiguration
     */
    public class AppSensorSecurityConfiguration : ConfigurationSection
    {
        #region ctor
        private static volatile AppSensorSecurityConfiguration singletonInstance;

        private AppSensorSecurityConfiguration() { }

        public static AppSensorSecurityConfiguration GetInstance()
        {
            if (singletonInstance == null)
            {
                lock (typeof(AppSensorSecurityConfiguration))
                {
                    if (singletonInstance == null)
                    {
                        singletonInstance = ConfigurationManager.GetSection("appsensor") as AppSensorSecurityConfiguration;;
                    }
                }
            }
            return singletonInstance;
        }
        #endregion

        #region Events collection Property

        /// <summary>
		/// The XML name of the <see cref="IntrusionDetection"/> property.
		/// </summary>
		internal const string EventsPropertyName = "events";

		/// <summary>
		/// Gets or sets the IntrusionDetection.
		/// </summary>
        [ConfigurationProperty(EventsPropertyName, IsRequired = true, IsKey = false, IsDefaultCollection = true)]
        [ConfigurationCollection(typeof(EventElementCollection), AddItemName="event")]
		public EventElementCollection Events
		{
			get
			{
                return (EventElementCollection)this[EventsPropertyName];
			}
			set
			{
                this[EventsPropertyName] = value;
			}
		}

		#endregion

        #region AppSensor properties
        internal const string REDIRECT_URL_IMPLEMENTATION = "redirecturl";
        /// <summary>
        /// The url to use if redirect is the required action
        /// </summary>
        [ConfigurationProperty(REDIRECT_URL_IMPLEMENTATION)]
        public string RedirectUrl
        {
            get
            {
                return (string)this[REDIRECT_URL_IMPLEMENTATION];
            }
            set
            {
                this[REDIRECT_URL_IMPLEMENTATION] = value;
            }
        }

        internal const string SMS_ADMIN_IMPLEMENTATION = "smsAdmin";
        /// <summary>
        /// The details to use when sms-ing the admin
        /// </summary>
        [ConfigurationProperty(SMS_ADMIN_IMPLEMENTATION)]
        public EmailSmsElement SmsAdmin
        {
            get
            {
                return (EmailSmsElement)this[SMS_ADMIN_IMPLEMENTATION];
            }
            set
            {
                this[SMS_ADMIN_IMPLEMENTATION] = value;
            }
        }

        internal const string EMAIL_ADMIN_IMPLEMENTATION = "emailAdmin";
        /// <summary>
        /// The details to use when email-ing the admin
        /// </summary>
        [ConfigurationProperty(EMAIL_ADMIN_IMPLEMENTATION, DefaultValue = "")]
        public EmailSmsElement EmailAdmin
        {
            get
            {
                return (EmailSmsElement)this[EMAIL_ADMIN_IMPLEMENTATION];
            }
            set
            {
                this[EMAIL_ADMIN_IMPLEMENTATION] = value;
            }
        }

        internal const string DISABLE_COMPONENT_EXTENSIONS_TO_CHECK_IMPLEMENTATION = "disablecomponentextensionstocheck";
        /// <summary>
        /// A comma-separated list of component extensions to check
        /// </summary>
        [ConfigurationProperty(DISABLE_COMPONENT_EXTENSIONS_TO_CHECK_IMPLEMENTATION, DefaultValue = "")]
        public string DisableComponentExtensionsToCheck
        {
            get
            {
                return (string)this[DISABLE_COMPONENT_EXTENSIONS_TO_CHECK_IMPLEMENTATION];
            }
            set
            {
                this[DISABLE_COMPONENT_EXTENSIONS_TO_CHECK_IMPLEMENTATION] = value;
            }
        }
        
        internal const string SQL_INJECTION_ATTACK_PATTERNS_IMPLEMENTATION = "sqlinjectionattackpatterns";
        /// <summary>
        /// A comma-separated list of sql injection attack patterns
        /// </summary>
        [ConfigurationProperty(SQL_INJECTION_ATTACK_PATTERNS_IMPLEMENTATION, DefaultValue = "")]
        public string SqlInjectionAttackPatterns
        {
            get
            {
                return (string)this[SQL_INJECTION_ATTACK_PATTERNS_IMPLEMENTATION];
            }
            set
            {
                this[SQL_INJECTION_ATTACK_PATTERNS_IMPLEMENTATION] = value;
            }
        }

        internal const string XSS_ATTACK_PATTERNS_IMPLEMENTATION = "xssattackpatterns";
        /// <summary>
        /// A comma-separated list of xss attack patterns
        /// </summary>
        [ConfigurationProperty(XSS_ATTACK_PATTERNS_IMPLEMENTATION, DefaultValue = "")]
        public string XssAttackPatterns
        {
            get
            {
                return (string)this[XSS_ATTACK_PATTERNS_IMPLEMENTATION];
            }
            set
            {
                this[XSS_ATTACK_PATTERNS_IMPLEMENTATION] = value;
            }
        }

        internal const string DISABLE_COMPONENT_EXCEPTIONS_IMPLEMENTATION = "disablecomponentexceptions";
        /// <summary>
        /// Any components that should not be disabled even if a rule requires it
        /// </summary>
        [ConfigurationProperty(DISABLE_COMPONENT_EXCEPTIONS_IMPLEMENTATION, DefaultValue = "")]
        public string DisableComponentExceptions
        {
            get
            {
                return (string)this[DISABLE_COMPONENT_EXCEPTIONS_IMPLEMENTATION];
            }
            set
            {
                this[DISABLE_COMPONENT_EXCEPTIONS_IMPLEMENTATION] = value;
            }
        }

        internal const string INTRUSION_STORE_IMPLEMENTATION = "intrusionStore";
        /// <summary>
        /// Gets or sets the name of the intrusion store type
        /// </summary>
        [ConfigurationProperty(INTRUSION_STORE_IMPLEMENTATION, DefaultValue = "org.owasp.appsensor.intrusiondetection.reference.DefaultIntrusionStore")]
        public string IntrusionStore
        {
            get
            {
                return (string)this[INTRUSION_STORE_IMPLEMENTATION];
            }
            set
            {
                this[INTRUSION_STORE_IMPLEMENTATION] = value;
            }
        }

        internal const string RESPONSE_ACTION_IMPLEMENTATION = "responseAction";
        /// <summary>
        /// Gets or sets the type name to use for response actions
        /// </summary>
        [ConfigurationProperty(RESPONSE_ACTION_IMPLEMENTATION, DefaultValue = "org.owasp.appsensor.intrusiondetection.reference.DefaultResponseAction")]
        public string ResponseAction
        {
            get
            {
                return (string)this[RESPONSE_ACTION_IMPLEMENTATION];
            }
            set
            {
                this[RESPONSE_ACTION_IMPLEMENTATION] = value;
            }
        }

        internal const string ALL_HTTP_METHODS = "allHttpMethods";
        /// <summary>
        /// Gets or sets the comma-separated list of all http verbs
        /// </summary>
        [ConfigurationProperty(ALL_HTTP_METHODS, DefaultValue = "GET,POST,PUT,DELETE,OPTIONS,HEAD,PUT,DELETE,TRACE,CONNECT")]
        public string AllHttpMethods
        {
            get
            {
                return (string)this[ALL_HTTP_METHODS];
            }
            set
            {
                this[ALL_HTTP_METHODS] = value;
            }
        }

        internal const string VALID_HTTP_METHODS = "validHttpMethods";
        /// <summary>
        /// Gets or sets the http methods that are expected in this app
        /// </summary>
        [ConfigurationProperty(VALID_HTTP_METHODS, IsRequired = true)]
        public string ValidHttpMethods
        {
            get
            {
                return (string)this[VALID_HTTP_METHODS];
            }
            set
            {
                this[VALID_HTTP_METHODS] = value;
            }
        }

        internal const string ASUTILITIES_IMPLEMENTATION = "asUtilities";
        /// <summary>
        /// Gets or sets the type name of the AppSensor utilities class
        /// </summary>
        [ConfigurationProperty(ASUTILITIES_IMPLEMENTATION, DefaultValue = "org.owasp.appsensor.reference.DefaultASUtilities")]
        public string AsUtilities
        {
            get
            {
                return (string)this[ASUTILITIES_IMPLEMENTATION];
            }
            set
            {
                this[ASUTILITIES_IMPLEMENTATION] = value;
            }
        }

        internal const string SERVICESTORE_IMPLEMENTATION = "serviceStore";
        /// <summary>
        /// Gets or sets the type name of the AppSensor utilities class
        /// </summary>
        [ConfigurationProperty(SERVICESTORE_IMPLEMENTATION, DefaultValue = "org.owasp.appsensor.reference.DefaultServiceStore")]
        public string ServiceStore
        {
            get
            {
                return (string)this[SERVICESTORE_IMPLEMENTATION];
            }
            set
            {
                this[SERVICESTORE_IMPLEMENTATION] = value;
            }
        }

        internal const string TRENDLOGGER_IMPLEMENTATION = "trendLogger";
        /// <summary>
        /// Gets or sets the type name of the AppSensor utilities class
        /// </summary>
        [ConfigurationProperty(TRENDLOGGER_IMPLEMENTATION, DefaultValue = "org.owasp.appsensor.trendmonitoring.reference.InMemoryTrendLogger")]
        public string TrendLogger
        {
            get
            {
                return (string)this[TRENDLOGGER_IMPLEMENTATION];
            }
            set
            {
                this[TRENDLOGGER_IMPLEMENTATION] = value;
            }
        }
        #endregion

        /** Name of encrypted properties lookup string that should be in appsensor.properties file */
        public const String APPSENSOR_ENCRYPTED_PROPERTIES_FILE_LOOKUP_KEY = "appsensor.encrypted.properties.file";


        /**
         * This method is not implemented in this class.  GetAppSensorQuota() should be used instead.
         * @param eventName event name
         * @return Threshold object
         */
        public Threshold GetQuota(String eventName)
        {
            throw new NotImplementedException("This method is not implemented in this class.");
        }

        /**
         * This method retrieves the configuration information from the Esapi.properties file
         * for managing when and how to respond to an intrusion or set of intrusions.
         * @param eventName event code to lookup
         * @return AppSensorThreshold object containing intrusion detection configuration information
         */
        public AppSensorThreshold GetAppSensorQuota(String eventName)
        {
            if (Events[eventName] == null)
                return null;

            var count = Events[eventName].Interval;
            var interval = Events[eventName].Interval;
            var actions = Events[eventName].Actions.ToList();
            var disableComponentDuration = Events[eventName].DisableComponentDuration;
            var disableComponentTimeScale = Events[eventName].DisableComponentTimeScale;
            var disableComponentForUserDuration = Events[eventName].DisableComponentForUserDuration;
            var disableComponentForUserTimeScale = Events[eventName].DisableComponentForUserTimeScale;

            if (count > 0 && interval > 0 && actions.Count > 0)
            {
                return new AppSensorThreshold(eventName, count, interval, actions,
                        disableComponentDuration, disableComponentTimeScale,
                        disableComponentForUserDuration, disableComponentForUserTimeScale);
            }
            return null;
        }
    }
}